const API = "http://127.0.0.1:5000";
const YEAR = "2026";
const today = new Date().toISOString().split("T")[0];

let settings = {};
let editingTransactionId = null;

const pageTitles = {
    "dashboard": "Dashboard",
    "transactions": "İşlemler",
    "settings": "Ayarlar",
    "reports": "Raporlar"
};

function showPage(id, buttonElement) {
    // Tüm sayfaları gizle
    document.querySelectorAll(".page").forEach(p => p.classList.remove("active"));
    document.getElementById(id).classList.add("active");

    // Tüm menü butonlarından active class'ını kaldır
    document.querySelectorAll(".sidebar button").forEach(btn => btn.classList.remove("active"));

    // Aktif butona active class'ı ekle
    if (buttonElement) {
        buttonElement.classList.add("active");
    } else {
        // Eğer buttonElement yoksa, butonu bul
        const buttons = document.querySelectorAll(".sidebar button");
        buttons.forEach(btn => {
            if (btn.textContent.trim() === pageTitles[id]) {
                btn.classList.add("active");
            }
        });
    }

    // Page title'ı güncelle
    const pageTitle = document.getElementById("page-title");
    if (pageTitle && pageTitles[id]) {
        pageTitle.textContent = pageTitles[id];
    }
}

function formatDate(dateString) {
    const date = new Date(dateString);
    const options = { year: 'numeric', month: 'long', day: 'numeric' };
    return date.toLocaleDateString('tr-TR', options);
}

async function init() {
    document.getElementById("t-date").value = today;
    document.getElementById("trans-date").value = today;
    document.getElementById("t-note-count").textContent = "0/200";

    // Bugünün tarihini göster
    document.getElementById("today-date").textContent = formatDate(today);

    // İlk sayfa olarak dashboard'u aktif yap
    showPage('dashboard');

    await loadSettings();
    await loadDashboard();
    await loadTransactions();
}

async function loadSettings() {
    try {
        const res = await fetch(`${API}/settings`);
        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.error || "Ayarlar yüklenemedi");
        }
        settings = await res.json();

        fillSettings();
        fillTransactionSelects();
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    }
}

function fillSettings() {
    fillList("income", "income-list");
    fillList("expense", "expense-list");
    fillList("payment-method", "method-list");
}

function fillList(type, elementId) {
    const ul = document.getElementById(elementId);
    ul.innerHTML = "";

    Object.entries(settings).forEach(([id, s]) => {
        if (s.type === type) {
            s.data.forEach((item, index) => {
                const li = document.createElement("li");
                li.style.display = "flex";
                li.style.justifyContent = "space-between";
                li.style.alignItems = "center";
                li.style.marginBottom = "8px";

                const span = document.createElement("span");
                span.textContent = item;
                li.appendChild(span);

                const deleteBtn = document.createElement("button");
                deleteBtn.className = "btn-delete-small";
                deleteBtn.textContent = "Sil";
                deleteBtn.onclick = () => deleteSettingItem(id, type, index);
                li.appendChild(deleteBtn);

                ul.appendChild(li);
            });
        }
    });
}

async function addSettingItem(type) {
    try {
        const inputId = type === "income" ? "income-input" : type === "expense" ? "expense-input" : "method-input";
        const input = document.getElementById(inputId);
        const value = input.value.trim();

        if (!value) {
            showToast("Lütfen bir değer girin", "error");
            return;
        }

        // Mevcut ayarı bul
        const settingEntry = Object.entries(settings).find(([id, s]) => s.type === type);
        if (!settingEntry) {
            showToast("Ayar bulunamadı", "error");
            return;
        }

        const [settingId, setting] = settingEntry;

        // Zaten varsa ekleme
        if (setting.data.includes(value)) {
            showToast("Bu öğe zaten mevcut", "error");
            input.value = "";
            return;
        }

        // Yeni öğeyi ekle
        const updatedData = [...setting.data, value];

        const res = await fetch(`${API}/settings/${settingId}`, {
            method: "PUT",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: updatedData })
        });

        if (!res.ok) {
            throw new Error("Ayar güncellenemedi");
        }

        input.value = "";
        showToast("Öğe başarıyla eklendi", "success");
        await loadSettings();
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    }
}

async function deleteSettingItem(settingId, type, index) {
    try {
        if (!confirm("Bu öğeyi silmek istediğinizden emin misiniz?")) {
            return;
        }

        const setting = settings[settingId];
        if (!setting || setting.type !== type) {
            showToast("Ayar bulunamadı", "error");
            return;
        }

        const updatedData = setting.data.filter((_, i) => i !== index);

        const res = await fetch(`${API}/settings/${settingId}`, {
            method: "PUT",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ data: updatedData })
        });

        if (!res.ok) {
            throw new Error("Ayar güncellenemedi");
        }

        showToast("Öğe başarıyla silindi", "success");
        await loadSettings();
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    }
}

function fillTransactionSelects() {
    const type = document.getElementById("t-type").value;
    const cat = document.getElementById("t-category");
    const method = document.getElementById("t-method");

    cat.innerHTML = "";
    method.innerHTML = "";

    Object.values(settings).forEach(s => {
        if (s.type === type) {
            s.data.forEach(i => cat.add(new Option(i, i)));
        }
        if (s.type === "payment-method") {
            s.data.forEach(i => method.add(new Option(i, i)));
        }
    });
}

document.getElementById("t-type").addEventListener("change", fillTransactionSelects);
document.getElementById("e-type").addEventListener("change", fillEditTransactionSelects);

document.getElementById("t-note").addEventListener("input", function () {
    document.getElementById("t-note-count").textContent = this.value.length + "/200";
});

function fillEditTransactionSelects() {
    const type = document.getElementById("e-type").value;
    const cat = document.getElementById("e-category");
    const method = document.getElementById("e-method");

    cat.innerHTML = "";
    method.innerHTML = "";

    Object.values(settings).forEach(s => {
        if (s.type === type) {
            s.data.forEach(i => cat.add(new Option(i, i)));
        }
        if (s.type === "payment-method") {
            s.data.forEach(i => method.add(new Option(i, i)));
        }
    });
}

async function addTransaction() {
    const submitBtn = document.querySelector("#transactions button[onclick='addTransaction()']");
    const originalText = submitBtn.textContent;

    try {
        // Form validasyonu
        const date = document.getElementById("t-date").value;
        const category = document.getElementById("t-category").value;
        const method = document.getElementById("t-method").value;
        const amount = document.getElementById("t-amount").value;

        if (!date) {
            showToast("Lütfen tarih seçin", "error");
            return;
        }
        if (!category) {
            showToast("Lütfen kategori seçin", "error");
            return;
        }
        if (!method) {
            showToast("Lütfen ödeme yöntemi seçin", "error");
            return;
        }
        if (!amount || Number(amount) <= 0) {
            showToast("Lütfen geçerli bir tutar girin", "error");
            return;
        }

        submitBtn.disabled = true;
        submitBtn.textContent = "Kaydediliyor...";

        const payload = {
            date: date,
            type: document.getElementById("t-type").value,
            category: category,
            method: method,
            amount: Number(amount),
            note: document.getElementById("t-note").value
        };

        const res = await fetch(`${API}/transactions?year=${YEAR}`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(payload)
        });

        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.error || "İşlem eklenemedi");
        }

        // Formu temizle
        document.getElementById("t-amount").value = "";
        document.getElementById("t-note").value = "";

        showToast("İşlem başarıyla eklendi", "success");
        await loadDashboard();
        await loadTransactions();
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    } finally {
        submitBtn.disabled = false;
        submitBtn.textContent = originalText;
    }
}

function openEditModal(transactionId, transaction) {
    editingTransactionId = transactionId;

    document.getElementById("e-date").value = transaction.date || "";
    document.getElementById("e-type").value = transaction.type || "income";
    document.getElementById("e-amount").value = transaction.amount || "";
    document.getElementById("e-note").value = transaction.note || "";

    // created_at bilgisini göster
    const createdAtInfo = document.getElementById("edit-created-at");
    if (transaction.created_at) {
        const createdDate = new Date(transaction.created_at);
        const formattedDate = createdDate.toLocaleDateString('tr-TR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
        createdAtInfo.textContent = `Şu tarihte oluşturuldu: ${formattedDate}`;
        createdAtInfo.style.display = "block";
    } else {
        createdAtInfo.style.display = "none";
    }

    fillEditTransactionSelects();

    // Kategori ve method değerlerini set et
    setTimeout(() => {
        if (transaction.category) {
            document.getElementById("e-category").value = transaction.category;
        }
        if (transaction.method) {
            document.getElementById("e-method").value = transaction.method;
        }
    }, 100);

    document.getElementById("edit-modal").classList.add("active");
}

function closeEditModal() {
    document.getElementById("edit-modal").classList.remove("active");
    editingTransactionId = null;
}

async function saveTransaction() {
    if (!editingTransactionId) return;

    const saveBtn = document.querySelector(".modal-footer button:last-child");
    const originalText = saveBtn.textContent;

    try {
        // Form validasyonu
        const date = document.getElementById("e-date").value;
        const category = document.getElementById("e-category").value;
        const method = document.getElementById("e-method").value;
        const amount = document.getElementById("e-amount").value;

        if (!date) {
            showToast("Lütfen tarih seçin", "error");
            return;
        }
        if (!category) {
            showToast("Lütfen kategori seçin", "error");
            return;
        }
        if (!method) {
            showToast("Lütfen ödeme yöntemi seçin", "error");
            return;
        }
        if (!amount || Number(amount) <= 0) {
            showToast("Lütfen geçerli bir tutar girin", "error");
            return;
        }

        saveBtn.disabled = true;
        saveBtn.textContent = "Kaydediliyor...";

        const payload = {
            date: date,
            type: document.getElementById("e-type").value,
            category: category,
            method: method,
            amount: Number(amount),
            note: document.getElementById("e-note").value
        };

        const res = await fetch(`${API}/transactions/${editingTransactionId}?year=${YEAR}`, {
            method: "PUT",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(payload)
        });

        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.error || "İşlem güncellenemedi");
        }

        closeEditModal();
        showToast("İşlem başarıyla güncellendi", "success");
        await loadDashboard();
        await loadTransactions();
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    } finally {
        saveBtn.disabled = false;
        saveBtn.textContent = originalText;
    }
}

async function deleteTransaction(transactionId) {
    if (!confirm("Bu işlemi silmek istediğinizden emin misiniz?")) {
        return;
    }

    try {
        const res = await fetch(`${API}/transactions/${transactionId}?year=${YEAR}`, {
            method: "DELETE"
        });

        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.error || "İşlem silinemedi");
        }

        showToast("İşlem başarıyla silindi", "success");
        await loadDashboard();
        await loadTransactions();
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    }
}

async function loadTransactions() {
    const tbody = document.getElementById("transaction-table");
    tbody.innerHTML = "<tr><td colspan='7' style='text-align: center; color: var(--text-soft);'>Yükleniyor...</td></tr>";

    try {
        const selectedDate = document.getElementById("trans-date").value || today;
        const res = await fetch(`${API}/transactions/range?year=${YEAR}&start=${selectedDate}&end=${selectedDate}`);
        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.error || "İşlemler yüklenemedi");
        }
        const data = await res.json();

        tbody.innerHTML = "";

        if (Object.keys(data).length === 0) {
            const tr = document.createElement("tr");
            tr.innerHTML = `<td colspan="7" style="text-align: center; color: var(--text-soft);">Bu tarihe ait işlem bulunamadı</td>`;
            tbody.appendChild(tr);
            return;
        }

        Object.entries(data).forEach(([id, t]) => {
            const tr = document.createElement("tr");

            // Gelir/gider durumuna göre renk class'ı ekle
            if (t.type === "income") {
                tr.classList.add("row-income");
            } else if (t.type === "expense") {
                tr.classList.add("row-expense");
            }

            const editBtn = document.createElement("button");
            editBtn.className = "btn-edit";
            editBtn.textContent = "Düzenle";
            editBtn.onclick = () => openEditModal(id, t);

            const deleteBtn = document.createElement("button");
            deleteBtn.className = "btn-delete";
            deleteBtn.textContent = "Sil";
            deleteBtn.onclick = () => deleteTransaction(id);

            const note = t.note || "";
            const noteDisplay = note.length > 50 ? note.slice(0, 50) + "…" : (note || "-");
            const noteEscaped = noteDisplay.replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;');
            const noteTitleAttr = note.length > 50 ? note.replace(/"/g, '&quot;') : "";
            tr.innerHTML = `
                <td>${t.date || "-"}</td>
                <td>${t.type === "income" ? "Gelir" : "Gider"}</td>
                <td>${t.category || "-"}</td>
                <td>${t.method || "-"}</td>
                <td>${t.amount || 0}</td>
                <td class="note-cell"${noteTitleAttr ? ` title="${noteTitleAttr}"` : ""}>${noteEscaped}</td>
                <td></td>
    `;
            const actionsCell = tr.querySelector("td:last-child");
            actionsCell.appendChild(editBtn);
            actionsCell.appendChild(deleteBtn);
            tbody.appendChild(tr);
        });
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    }
}

async function loadDashboard() {
    try {
        // Dashboard sadece bugünün tarihi için çalışır
        const res = await fetch(`${API}/summary/daily?year=${YEAR}&date=${today}`);
        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.error || "Dashboard verileri yüklenemedi");
        }
        const d = await res.json();

        document.getElementById("dash-income").textContent = d.income || 0;
        document.getElementById("dash-expense").textContent = d.expense || 0;
        document.getElementById("dash-profit").textContent = d.profit || 0;

        // Detay chart'ları render et
        renderReportDetails(
            d.income_details || {},
            d.expense_details || {},
            d.income || 0,
            d.expense || 0,
            "dash-income-chart",
            "dash-expense-chart"
        );
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    }
}

async function loadReport() {
    const reportBtn = document.querySelector("#reports button[onclick='loadReport()']");
    const originalText = reportBtn.textContent;

    try {
        const start = document.getElementById("r-start").value;
        const end = document.getElementById("r-end").value;

        if (!start || !end) {
            showToast("Lütfen başlangıç ve bitiş tarihlerini seçin", "error");
            return;
        }

        if (start > end) {
            showToast("Başlangıç tarihi bitiş tarihinden sonra olamaz", "error");
            return;
        }

        reportBtn.disabled = true;
        reportBtn.textContent = "Yükleniyor...";

        const res = await fetch(`${API}/summary/range?year=${YEAR}&start=${start}&end=${end}`);
        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.error || "Rapor yüklenemedi");
        }
        const d = await res.json();

        document.getElementById("rep-income").textContent = d.income || 0;
        document.getElementById("rep-expense").textContent = d.expense || 0;
        document.getElementById("rep-profit").textContent = d.profit || 0;

        // Detay chart'ları render et
        renderReportDetails(
            d.income_details || {},
            d.expense_details || {},
            d.income || 0,
            d.expense || 0,
            "income-chart",
            "expense-chart"
        );

        showToast("Rapor başarıyla yüklendi", "success");
    } catch (error) {
        showToast("Hata: " + error.message, "error");
    } finally {
        reportBtn.disabled = false;
        reportBtn.textContent = originalText;
    }
}

function renderReportDetails(incomeDetails, expenseDetails, totalIncome, totalExpense, incomeChartId = "income-chart", expenseChartId = "expense-chart") {
    const incomeChart = document.getElementById(incomeChartId);
    const expenseChart = document.getElementById(expenseChartId);

    // Gelir detaylarını render et
    incomeChart.innerHTML = "";
    if (Object.keys(incomeDetails).length === 0) {
        incomeChart.innerHTML = '<div class="chart-empty">Gelir detayı bulunamadı</div>';
    } else {
        // Kategorileri tutara göre sırala (büyükten küçüğe)
        const sortedIncome = Object.entries(incomeDetails)
            .sort((a, b) => b[1] - a[1]);

        sortedIncome.forEach(([category, amount]) => {
            const percentage = totalIncome > 0 ? ((amount / totalIncome) * 100).toFixed(1) : 0;
            const chartItem = createChartItem(category, amount, percentage, "income");
            incomeChart.appendChild(chartItem);
        });
    }

    // Gider detaylarını render et
    expenseChart.innerHTML = "";
    if (Object.keys(expenseDetails).length === 0) {
        expenseChart.innerHTML = '<div class="chart-empty">Gider detayı bulunamadı</div>';
    } else {
        // Kategorileri tutara göre sırala (büyükten küçüğe)
        const sortedExpense = Object.entries(expenseDetails)
            .sort((a, b) => b[1] - a[1]);

        sortedExpense.forEach(([category, amount]) => {
            const percentage = totalExpense > 0 ? ((amount / totalExpense) * 100).toFixed(1) : 0;
            const chartItem = createChartItem(category, amount, percentage, "expense");
            expenseChart.appendChild(chartItem);
        });
    }
}

function createChartItem(category, amount, percentage, type) {
    const item = document.createElement("div");
    item.className = `chart-item chart-item-${type}`;

    const formattedAmount = new Intl.NumberFormat('tr-TR', {
        style: 'currency',
        currency: 'TRY',
        minimumFractionDigits: 0,
        maximumFractionDigits: 0
    }).format(amount);

    // Yüzde metnini bar içinde veya dışında göstermek için kontrol
    const percentageText = percentage > 5 ? `<span class="chart-item-percentage">${percentage}%</span>` : '';

    item.innerHTML = `
        <div class="chart-item-header">
            <span class="chart-item-label">${category}</span>
            <div class="chart-item-meta">
                <span class="chart-item-amount">${formattedAmount}</span>
                ${percentage <= 5 ? `<span class="chart-item-percentage-outside">${percentage}%</span>` : ''}
            </div>
        </div>
        <div class="chart-item-bar-container">
            <div class="chart-item-bar" style="width: ${percentage}%">
                ${percentageText}
            </div>
        </div>
    `;

    return item;
}

function showToast(message, type = "info") {
    const toast = document.getElementById("toast");
    toast.textContent = message;
    toast.className = `toast ${type}`;
    toast.classList.add("show");

    setTimeout(() => {
        toast.classList.remove("show");
    }, 3000);
}

init();